'use client';

import { useParams, useRouter } from 'next/navigation';
import { Card, CardContent, CardHeader, CardTitle, CardFooter } from '@/components/ui/card';
import { OrderCard } from '@/components/order-card';
import { Alert, AlertDescription, AlertTitle } from '@/components/ui/alert';
import { Truck, ArrowLeft } from 'lucide-react';
import { Button } from '@/components/ui/button';
import Link from 'next/link';
import { useDoc, useFirestore, useUser, useMemoFirebase } from '@/firebase';
import { doc } from 'firebase/firestore';
import type { Order } from '@/lib/data';
import { Skeleton } from '@/components/ui/skeleton';

export default function OrderDetailsPage() {
  const params = useParams();
  const router = useRouter();
  const orderId = params.orderId as string;
  const { user } = useUser();
  const firestore = useFirestore();

  const orderDocRef = useMemoFirebase(
    () => (user ? doc(firestore, 'users', user.uid, 'orders', orderId) : null),
    [firestore, user, orderId]
  );
  
  const { data: order, isLoading, error } = useDoc<Order>(orderDocRef);
  
  if (isLoading) {
    return (
      <div className="p-4 md:p-8">
        <Skeleton className="h-8 w-48 mb-4" />
        <Skeleton className="h-64 w-full" />
      </div>
    );
  }

  return (
    <div className="p-4 md:p-8">
      <div className="flex items-center mb-4">
        <Button variant="ghost" size="icon" onClick={() => router.back()} className="mr-2">
            <ArrowLeft className="h-4 w-4" />
        </Button>
        <h1 className="text-2xl font-bold">Order Details</h1>
      </div>
      
      {order ? (
        <OrderCard order={order} />
      ) : (
        <Card>
          <CardContent className="pt-6">
            <Alert variant="destructive">
              <Truck className="h-4 w-4" />
              <AlertTitle>Order Not Found</AlertTitle>
              <AlertDescription>
                The order you are looking for does not exist or you do not have permission to view it.
              </AlertDescription>
            </Alert>
          </CardContent>
          <CardFooter>
              <Button asChild variant="outline">
                  <Link href="/dashboard/orders">Back to Orders</Link>
              </Button>
          </CardFooter>
        </Card>
      )}
    </div>
  );
}
